/**
 *******************************************************************************
 *
 * @file        API_Flash.c
 *
 * @brief       Application for serial flash 
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2022/09/26
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_SPI_MID.h"
#include "MG32_DMA_MID.h"
#include "MG32_GPIO_MID.h"

#ifndef _API_Flash_H
/*!< _API_Flash_H */ 
#define _API_Flash_H

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/**
 * @name    ========MG32 Peripheral Define ==============
 *
 */ 
///@{
#define Flash_nCS       PD0         /*!< Define software nCS Pin.               */
///@}

/**
 * @name    ============== MX25R512F Command ==============
 *
 */ 
///@{
#define FLASH_NORMAL_READ           0x03    /*!< Read data bytes                                        */
#define FLASH_FAST_READ             0x0B    /*!< Read data bytes at higher speed                        */

#define FLASH_2IO_READ              0xBB    /*!< 2 x I/O read mode                                      */
#define FLASH_1I_2O_READ            0x3B    /*!< Dual read mode                                         */
#define FLASH_4IO_READ              0xEB    /*!< 4 x I/O read mode                                      */
#define FLASH_1I_4O_Read            0x6B    /*!< Quad read mode                                         */

#define FLASH_PAGE_PROGRAM          0x02    /*!< Page program                                           */
#define FLASH_QUADPAGE_PROGRAM      0x38    /*!< 4 x I/O page program                                   */

#define FLASH_SECTOR_ERASE          0x20    /*!< 1 sector = 4K bytes                                    */
#define FLASH_BLOCK32K_ERASE        0x52    /*!< Block erase 32K bytes                                  */
#define FLASH_BLOCK64K_ERASE        0xD8    /*!< Block erase 62K bytes                                  */
#define FLASH_CHIP_ERASE            0x60    /*!< Chip erase                                             */

#define FLASH_SFDP_READ             0x5A    /*!< Read SFDP mode                                         */
#define FLASH_WRITE_ENABLE          0x06    /*!< Write enable                                           */
#define FLASH_WRITE_DISABLE         0x04    /*!< Write disable                                          */
#define FLASH_READ_STATUS           0x05    /*!< Read Status Register                                   */
#define FLASH_READ_CONFIGURATION    0x15    /*!< Read configuration register                            */
#define FLASH_WRITE_STATUS          0x01    /*!< Write status register                                  */
#define FLASH_SUSPEND_PROGRAM       0x75    /*!< Suspends erase                                         */
#define FLASH_SUSPEND_ERASE         0xB0    /*!< Suapends erase                                         */
#define FLASH_RESUMES_PROGRAM       0x7A    /*!< Resume program                                         */
#define FLASH_RESUMES_REASE         0x30    /*!< Resume erase                                           */
#define FLASH_DEEP_POWER_DOWN       0xB9    /*!< Deep power-down                                        */
#define FLASH_SET_BURST_LENGTH      0xC0    /*!< Burst read                                             */
#define FLASH_READ_IDENTIFICATION   0x9F    /*!< Read manufacturer ID + memory type + memory density.   */
#define FLASH_READ_ELECTRONIC_ID    0xAB    /*!< Read electronic ID.                                    */
#define FLASH_READ_REMS             0x90    /*!< Read manufacturer & device ID.                         */
#define FLASH_ENTER_SECURED_OTP     0xB1    /*!< Enter secured OTP                                      */
#define FLASH_EXIT_SECURED_OTP      0xC1    /*!< Exit secured OTP                                       */
#define FLASH_READ_SECURITY_REG     0x2B    /*!< Read security register                                 */
#define FLASH_WRITE_SECURITY_REG    0x2F    /*!< Write security register                                */
#define FLASH_NOP                   0x00    /*!< Nop                                                    */
#define FLASH_RESET_ENABLE          0x66    /*!< Reset enable                                           */
#define FLAHS_RESET_MEMORY          0x99    /*!< Reset memory                                           */
///@}

/**
 * @name API_Error_Code 
 *      APi Error Code
 */ 
//!@{
#define MID_SPI_ERROR_NONE              (0x00000000U)   /*!< No error                               */
#define MID_SPI_ERROR_MOD               (0x00000001U)   /*!< MODF error                             */
#define MID_SPI_ERROR_CRC               (0x00000002U)   /*!< CRC error                              */
#define MID_SPI_ERROR_OVR               (0x00000004U)   /*!< ROVRF error                            */
#define MID_SPI_ERROR_WE                (0x00000008U)   /*!< WEF error                              */
#define MID_SPI_ERROR_DMA               (0x00000010U)   /*!< DMA transfer error                     */
#define MID_SPI_ERROR_FLAG              (0x00000020U)   /*!< Error on RXNE/TXE/BSY/FTLVL/FRLVL Flag */
#define MID_SPI_ERROR_ABORT             (0x00000040U)   /*!< Error during SPI Abort procedure       */
#define MID SPI_ERROR_URD               (0x00000080U)   /*!< TUDRF error                            */
//!@}

#define FLASH_nCS_ACTIVE        0                      /*!< Define polarity (nCS = Active).                     */
#define FLASH_nCS_NoACTIVE      1                      /*!< Define polarity (nCS = No Active).                  */ 

#define FLASH_MAXSIZE           64*1024                /*!< Flash max size.                                     */
#define FLASH_PROGRAM_MAX       256                    /*!< Define Flash program max bytes  in one write enable.*/ 

#define Flash_Dummy_Data        0xFFFFFFFF             /*!< Define DUMMY Data.                                  */

#define FLASH_RDID_CMP          0x1028C2               /*!< Manufacturer ID  + Memory type + Memory density.    */
#define FLASH_RES_CMP           0x10                   /*!< Electronic ID.                                      */
#define FLASH_REMS_CMP          0xC210                 /*!< Manufacturer ID + Device ID.                        */


#define MX25R512                0x00C22810              /*!< Flash MX25R512 ID 512K bits */
#define MX25R6435               0x00C22817              /*!< Flash MX25R6435 ID 64M bits */
#define S25FL256LAVF01          0x00016019              /*!< Flash S25FL256LAVF01 ID 256M bits */

#define FLASH_StatusRegister_WIP    0x01
#define FLASH_StatusRegister_WEL    0x02
#define FLASH_StatusRegister_BP30   0x3C
#define FLASH_StatusRegister_QE     0x40
#define FLASH_StatusRegister_SRWD   0x80

/* Private typedef -----------------------------------------------------------*/
/**
 * @name    ============== Flash control parameter ==============
 *  
 */ 
/*! @enum   Flash_ERASE_TypeDef
    @brief  The Flash Erase type.
*/
///@{
typedef enum
{
    FLASH_ERASE_SECTOR   = FLASH_SECTOR_ERASE,          /*!< Erase type = Sector(4K byte) erase.                  */
    FLASH_ERASE_BLOCK32K = FLASH_BLOCK32K_ERASE,        /*!< Erase type = 32K byte block erase.                   */
    FLASH_ERASE_BLOCK64K = FLASH_BLOCK64K_ERASE,        /*!< Erase type = 64K byte block erase.                   */
    FLASH_ERASE_CHIP     = FLASH_CHIP_ERASE,            /*!< Erase type = Chip erase.                             */
}Flash_ERASE_TypeDef;


/** 
 * @enum    API_StateTypeDef
 * @brief   API State structure definition
 */  
typedef enum
{
    API_FLASH_STATE_RESET      = 0x00U,                       /*!< Peripheral not Initialized                          */
    API_FLASH_STATE_READY      = 0x01U,                       /*!< Peripheral Initialized and ready for use            */
    API_FLASH_STATE_BUSY       = 0x02U,                       /*!< an internal process is ongoing                      */
    API_FLASH_STATE_READ       = 0x03U,                       /*!< Data Transmission process is ongoing                */
    API_FLASH_STATE_ERASE      = 0x04U,                       /*!< Data Reception process is ongoing                   */
    API_FLASH_STATE_PROGRAM    = 0x05U,                       /*!< Data Transmission and Reception process is ongoing  */
    API_FLASH_STATE_ERROR      = 0x06U,                       /*!< SPI error state                                     */
    API_FLASH_STATE_ABORT      = 0x07U                        /*!< SPI abort is ongoing                                */
}API_Flash_StateTypeDef;


/** 
  * @enum   API_LockTypeDef
  * @brief  API Lock structures definition  
  */
typedef enum 
{
    API_UNLOCKED = 0x00,                                /*!<*/
    API_LOCKED   = 0x01,                                /*!<*/
}API_LockTypeDef;
///@}

/*! @struct FlashCTR_TypeDef
    @brief  Flash data control.
*/ 
typedef struct
{
         uint8_t                 *pBuffer;               /*!< Pointer of program data buffer.                    */
         uint8_t                 *rBuffer;               /*!< Pointer of recevie buffer.                         */
         ctype                   Address;                /*!< Control Flash address.                             */
         uint32_t                ErrorCode;              /*!< SPI/URTx Error code.                               */
         uint32_t                Length;                 /*!< To control Flash read and write total length.      */
         MID_SPI_Struct          Peripheral;             /*!< SPIx/USPIx registers base address                  */
    __IO API_Flash_StateTypeDef  State;                  /*!< SPI/URTx communication state.                      */
         API_LockTypeDef         Lock;                   /*!< Locking object.                                    */
         uint8_t                 __Reserved[2];          /*!< Reserved byte.                                     */
}API_FlashHandlerTypeDef;

///*! @struct FlashCTR_TypeDef
//    @brief  Flash data control.
//*/ 
//typedef struct
//{
//    uint8_t               *pBuffer;                     /*!< Pointer of program data buffer.                      */
//    uint8_t               *rBuffer;                     /*!< Pointer of recevie buffer.                           */
//    ctype                 Address;                      /*!< Control Flash address.                               */
//    uint32_t              Total_Length;                 /*!< To control Flash read and write total address.       */
//    uint32_t              BLen;                         /*!< To control Flash write bytes in a time.              */
//    uint32_t              DataCnt;                      /*!< Use for Flash read and write operation parameter.    */
//}FlashCTR_TypeDef;


/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
uint32_t API_Flash_GetID (void);
void API_Flash_CheckBusy (void);
MID_StatusTypeDef API_Flsh_QuadEnable_Cmd (FunctionalState QE);

/**
 * @name    Function announce
 *  
 */ 
///@{ 
void API_Flash_Init (void);
MID_StatusTypeDef API_Flash_Erase (uint32_t Address, uint16_t ErasePage);
void API_Flash_MultiBytesWrite (uint32_t Address, uint8_t *DataSource, uint32_t Length);
void API_Flash_MultiBytesRead (uint32_t Address, uint8_t *BufferAddreass, uint32_t Length);
///@}  

/* Exported variables --------------------------------------------------------*/
extern API_FlashHandlerTypeDef     API_Flash;

/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/





#endif  // _API_Flash_H

